const RPC_URLS = {
    Ethereum: "https://rpc.ankr.com/eth",
    BSC: "https://bsc-dataseed.binance.org/",
    Polygon: "https://polygon-rpc.com/",
    Arbitrum: "https://arb1.arbitrum.io/rpc",
    Optimism: "https://mainnet.optimism.io"
};

// Hỗ trợ Dogecoin thông qua API BlockCypher
const DOGE_API = "https://api.blockcypher.com/v1/doge/main/addrs/";

const ERC20_TOKENS = {
    SHIBA: { address: "0x95aD61b0a150d79219dCF64E1E6Cc01f0B64C4cE", decimals: 18, chain: "Ethereum" }, // SHIBA trên Ethereum
    USDT: { address: "0xdAC17F958D2ee523a2206206994597C13D831ec7", decimals: 6, chain: "Ethereum" } // USDT trên Ethereum
};

function isValidSeed(seed) {
    try {
        ethers.Wallet.fromPhrase(seed);
        return true;
    } catch (e) {
        return false;
    }
}

async function getERC20Balance(provider, wallet, token) {
    try {
        const tokenContract = new ethers.Contract(
            token.address,
            ["function balanceOf(address) view returns (uint256)"],
            provider
        );
        const balance = await tokenContract.balanceOf(wallet);
        return ethers.formatUnits(balance, token.decimals);
    } catch (e) {
        return "Lỗi";
    }
}

async function getDogeBalance(address) {
    try {
        const response = await fetch(DOGE_API + address);
        const data = await response.json();
        return data.balance ? (data.balance / 1e8).toFixed(4) : "0";
    } catch (e) {
        return "Lỗi";
    }
}

async function checkSeeds() {
    const input = document.getElementById("seedInput").value.trim();
    const resultElement = document.getElementById("result");
    resultElement.textContent = "🔄 Đang kiểm tra...\n";

    if (!input) {
        resultElement.textContent = "⚠️ Vui lòng nhập ít nhất một seed phrase.";
        return;
    }

    const seeds = input.split("\n").map(s => s.trim()).filter(Boolean);

    for (let seed of seeds) {
        let result = "";
        try {
            if (!isValidSeed(seed)) {
                result = `❌ Seed không hợp lệ: ${seed}\n`;
                resultElement.textContent += result;
                continue;
            }

            const wallet = ethers.Wallet.fromPhrase(seed);
            const address = wallet.address;
            result = `✅ Seed hợp lệ: ${seed}\n🔹 Địa chỉ: ${address}\n`;

            for (const [chain, rpc] of Object.entries(RPC_URLS)) {
                const provider = new ethers.JsonRpcProvider(rpc);
                try {
                    const balance = await provider.getBalance(address);
                    const ethBalance = ethers.formatEther(balance);
                    result += `   🔸 ${chain}: ${ethBalance} ${chain === "BSC" ? "BNB" : "ETH"}\n`;
                } catch (e) {
                    result += `   ⚠️ Không thể kiểm tra trên ${chain}\n`;
                }
                resultElement.textContent += result;
            }

            // Kiểm tra số dư DOGE
            const dogeBalance = await getDogeBalance(address);
            result += `   🐶 Dogecoin: ${dogeBalance} DOGE\n`;
            resultElement.textContent += result;

            // Kiểm tra số dư ERC-20 (SHIBA, USDT)
            for (const [tokenName, token] of Object.entries(ERC20_TOKENS)) {
                if (RPC_URLS[token.chain]) {
                    const provider = new ethers.JsonRpcProvider(RPC_URLS[token.chain]);
                    const balance = await getERC20Balance(provider, address, token);
                    result += `   💰 ${tokenName}: ${balance}\n`;
                    resultElement.textContent += result;
                }
            }

            result += "\n";
            resultElement.textContent += result;
        } catch (error) {
            result = `⚠️ Lỗi với seed: ${seed} - ${error.message}\n`;
            resultElement.textContent += result;
        }
    }
}
